# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class IssueUpdater < IssueBase
          def update_existing_issue(issue, grouped_failure)
            log_issue_update(issue, grouped_failure)
            append_failures_to_issue(issue, grouped_failure[:failures])
            add_update_comment(issue, grouped_failure[:failures].size)
          end

          private

          def append_failures_to_issue(issue, failures)
            current_issue = @client.find_issues(iid: issue.iid).first
            return unless current_issue

            existing_description = current_issue.description
            affected_tests_match = existing_description.match(/### Affected Tests \((\d+) failures?\)/)
            return unless affected_tests_match

            updated_description = build_updated_description(existing_description, affected_tests_match, failures)
            update_issue_description(issue, updated_description, failures.size)
          end

          def update_issue_description(issue, updated_description, failure_count)
            handle_gitlab_api_error("updating issue", "##{issue.web_url}") do
              @client.edit_issue(iid: issue.iid, options: { description: updated_description })
              Runtime::Logger.info "Successfully appended #{failure_count} failures to issue #{issue.web_url}"
              true
            end
          end

          def build_updated_description(existing_description, affected_tests_match, failures)
            current_count = affected_tests_match[1].to_i
            new_count = current_count + failures.size

            updated_description = existing_description.gsub(
              /### Affected Tests \(\d+ failures?\)/,
              "### Affected Tests (#{new_count} failures)"
            )
            insert_new_failures(updated_description, failures)
          end

          def insert_new_failures(description, failures)
            formatter = IssueFormatter.new
            new_failure_entries = formatter.format_affected_tests(failures)

            test_section_end = description.index('### Pipeline Information')
            return description unless test_section_end

            insertion_point = description.rindex("\n", test_section_end - 1)
            return description unless insertion_point

            "#{description[0..insertion_point]}#{new_failure_entries}\n#{description[insertion_point + 1..]}"
          end

          def add_update_comment(issue, failure_count)
            pipeline_url = ENV['CI_PIPELINE_URL'] || "Pipeline #{ENV.fetch('CI_PIPELINE_ID', nil)}"
            comment = build_update_comment(pipeline_url, failure_count)
            add_comment_to_issue(issue, comment)
          end

          def build_update_comment(pipeline_url, failure_count)
            "🔄 **New failures added from #{pipeline_url}**\n\n" \
              "Added #{failure_count} additional test failures with the same error pattern."
          end

          def add_comment_to_issue(issue, comment)
            handle_gitlab_api_error("adding comment to issue", issue.web_url) do
              @client.create_issue_note(iid: issue.iid, note: comment)
              Runtime::Logger.info "Comment added successfully to issue #{issue.web_url}"
              true
            end
          end

          def log_issue_update(issue, grouped_failure)
            pipeline_id = ENV.fetch('CI_PIPELINE_ID', nil)
            Runtime::Logger.info "Updating existing issue ##{issue.iid} with #{grouped_failure[:failures].size} new failures from pipeline #{pipeline_id}"
          end
        end
      end
    end
  end
end
