/*
 * Copyright 2012 Red Hat Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors: Ben Skeggs
 */
#include "priv.h"
#include "chan.h"
#include "conn.h"
#include "head.h"
#include "dp.h"
#include "ior.h"
#include "outp.h"

#include <core/client.h>
#include <core/ramht.h>
#include <subdev/bios.h>
#include <subdev/bios/disp.h>
#include <subdev/bios/init.h>
#include <subdev/bios/pll.h>
#include <subdev/devinit.h>
#include <subdev/i2c.h>
#include <subdev/mmu.h>
#include <subdev/timer.h>

#include <nvif/class.h>
#include <nvif/unpack.h>

static void
nv50_pior_clock(struct nvkm_ior *pior)
{
	struct nvkm_device *device = pior->disp->engine.subdev.device;
	const u32 poff = nv50_ior_base(pior);

	nvkm_mask(device, 0x614380 + poff, 0x00000707, 0x00000001);
}

static int
nv50_pior_dp_links(struct nvkm_ior *pior, struct nvkm_i2c_aux *aux)
{
	int ret = nvkm_i2c_aux_lnk_ctl(aux, pior->dp.nr, pior->dp.bw, pior->dp.ef);
	if (ret)
		return ret;

	return 1;
}

static const struct nvkm_ior_func_dp
nv50_pior_dp = {
	.links = nv50_pior_dp_links,
};

static void
nv50_pior_power_wait(struct nvkm_device *device, u32 poff)
{
	nvkm_msec(device, 2000,
		if (!(nvkm_rd32(device, 0x61e004 + poff) & 0x80000000))
			break;
	);
}

static void
nv50_pior_power(struct nvkm_ior *pior, bool normal, bool pu, bool data, bool vsync, bool hsync)
{
	struct nvkm_device *device = pior->disp->engine.subdev.device;
	const u32  poff = nv50_ior_base(pior);
	const u32 shift = normal ? 0 : 16;
	const u32 state = 0x80000000 | (0x00000001 * !!pu) << shift;
	const u32 field = 0x80000000 | (0x00000101 << shift);

	nv50_pior_power_wait(device, poff);
	nvkm_mask(device, 0x61e004 + poff, field, state);
	nv50_pior_power_wait(device, poff);
}

void
nv50_pior_depth(struct nvkm_ior *ior, struct nvkm_ior_state *state, u32 ctrl)
{
	/* GF119 moves this information to per-head methods, which is
	 * a lot more convenient, and where our shared code expect it.
	 */
	if (state->head && state == &ior->asy) {
		struct nvkm_head *head = nvkm_head_find(ior->disp, __ffs(state->head));

		if (!WARN_ON(!head)) {
			struct nvkm_head_state *state = &head->asy;
			switch ((ctrl & 0x000f0000) >> 16) {
			case 6: state->or.depth = 30; break;
			case 5: state->or.depth = 24; break;
			case 2: state->or.depth = 18; break;
			case 0: state->or.depth = 18; break; /*XXX*/
			default:
				state->or.depth = 18;
				WARN_ON(1);
				break;
			}
		}
	}
}

static void
nv50_pior_state(struct nvkm_ior *pior, struct nvkm_ior_state *state)
{
	struct nvkm_device *device = pior->disp->engine.subdev.device;
	const u32 coff = pior->id * 8 + (state == &pior->arm) * 4;
	u32 ctrl = nvkm_rd32(device, 0x610b80 + coff);

	state->proto_evo = (ctrl & 0x00000f00) >> 8;
	state->rgdiv = 1;
	switch (state->proto_evo) {
	case 0: state->proto = TMDS; break;
	default:
		state->proto = UNKNOWN;
		break;
	}

	state->head = ctrl & 0x00000003;
	nv50_pior_depth(pior, state, ctrl);
}

static const struct nvkm_ior_func
nv50_pior = {
	.state = nv50_pior_state,
	.power = nv50_pior_power,
	.clock = nv50_pior_clock,
	.dp = &nv50_pior_dp,
};

int
nv50_pior_new(struct nvkm_disp *disp, int id)
{
	return nvkm_ior_new_(&nv50_pior, disp, PIOR, id, false);
}

int
nv50_pior_cnt(struct nvkm_disp *disp, unsigned long *pmask)
{
	struct nvkm_device *device = disp->engine.subdev.device;

	*pmask = (nvkm_rd32(device, 0x610184) & 0x70000000) >> 28;
	return 3;
}

static int
nv50_sor_bl_set(struct nvkm_ior *ior, int lvl)
{
	struct nvkm_device *device = ior->disp->engine.subdev.device;
	const u32 soff = nv50_ior_base(ior);
	u32 div = 1025;
	u32 val = (lvl * div) / 100;

	nvkm_wr32(device, 0x61c084 + soff, 0x80000000 | val);
	return 0;
}

static int
nv50_sor_bl_get(struct nvkm_ior *ior)
{
	struct nvkm_device *device = ior->disp->engine.subdev.device;
	const u32 soff = nv50_ior_base(ior);
	u32 div = 1025;
	u32 val;

	val  = nvkm_rd32(device, 0x61c084 + soff);
	val &= 0x000007ff;
	return ((val * 100) + (div / 2)) / div;
}

const struct nvkm_ior_func_bl
nv50_sor_bl = {
	.get = nv50_sor_bl_get,
	.set = nv50_sor_bl_set,
};

void
nv50_sor_clock(struct nvkm_ior *sor)
{
	struct nvkm_device *device = sor->disp->engine.subdev.device;
	const int  div = sor->asy.link == 3;
	const u32 soff = nv50_ior_base(sor);

	nvkm_mask(device, 0x614300 + soff, 0x00000707, (div << 8) | div);
}

static void
nv50_sor_power_wait(struct nvkm_device *device, u32 soff)
{
	nvkm_msec(device, 2000,
		if (!(nvkm_rd32(device, 0x61c004 + soff) & 0x80000000))
			break;
	);
}

void
nv50_sor_power(struct nvkm_ior *sor, bool normal, bool pu, bool data, bool vsync, bool hsync)
{
	struct nvkm_device *device = sor->disp->engine.subdev.device;
	const u32  soff = nv50_ior_base(sor);
	const u32 shift = normal ? 0 : 16;
	const u32 state = 0x80000000 | (0x00000001 * !!pu) << shift;
	const u32 field = 0x80000000 | (0x00000001 << shift);

	nv50_sor_power_wait(device, soff);
	nvkm_mask(device, 0x61c004 + soff, field, state);
	nv50_sor_power_wait(device, soff);

	nvkm_msec(device, 2000,
		if (!(nvkm_rd32(device, 0x61c030 + soff) & 0x10000000))
			break;
	);
}

void
nv50_sor_state(struct nvkm_ior *sor, struct nvkm_ior_state *state)
{
	struct nvkm_device *device = sor->disp->engine.subdev.device;
	const u32 coff = sor->id * 8 + (state == &sor->arm) * 4;
	u32 ctrl = nvkm_rd32(device, 0x610b70 + coff);

	state->proto_evo = (ctrl & 0x00000f00) >> 8;
	switch (state->proto_evo) {
	case 0: state->proto = LVDS; state->link = 1; break;
	case 1: state->proto = TMDS; state->link = 1; break;
	case 2: state->proto = TMDS; state->link = 2; break;
	case 5: state->proto = TMDS; state->link = 3; break;
	default:
		state->proto = UNKNOWN;
		break;
	}

	state->head = ctrl & 0x00000003;
}

static const struct nvkm_ior_func
nv50_sor = {
	.state = nv50_sor_state,
	.power = nv50_sor_power,
	.clock = nv50_sor_clock,
	.bl = &nv50_sor_bl,
};

static int
nv50_sor_new(struct nvkm_disp *disp, int id)
{
	return nvkm_ior_new_(&nv50_sor, disp, SOR, id, false);
}

int
nv50_sor_cnt(struct nvkm_disp *disp, unsigned long *pmask)
{
	struct nvkm_device *device = disp->engine.subdev.device;

	*pmask = (nvkm_rd32(device, 0x610184) & 0x03000000) >> 24;
	return 2;
}

static void
nv50_dac_clock(struct nvkm_ior *dac)
{
	struct nvkm_device *device = dac->disp->engine.subdev.device;
	const u32 doff = nv50_ior_base(dac);

	nvkm_mask(device, 0x614280 + doff, 0x07070707, 0x00000000);
}

int
nv50_dac_sense(struct nvkm_ior *dac, u32 loadval)
{
	struct nvkm_device *device = dac->disp->engine.subdev.device;
	const u32 doff = nv50_ior_base(dac);

	dac->func->power(dac, false, true, false, false, false);

	nvkm_wr32(device, 0x61a00c + doff, 0x00100000 | loadval);
	mdelay(9);
	udelay(500);
	loadval = nvkm_mask(device, 0x61a00c + doff, 0xffffffff, 0x00000000);

	dac->func->power(dac, false, false, false, false, false);
	if (!(loadval & 0x80000000))
		return -ETIMEDOUT;

	return (loadval & 0x38000000) >> 27;
}

static void
nv50_dac_power_wait(struct nvkm_device *device, const u32 doff)
{
	nvkm_msec(device, 2000,
		if (!(nvkm_rd32(device, 0x61a004 + doff) & 0x80000000))
			break;
	);
}

void
nv50_dac_power(struct nvkm_ior *dac, bool normal, bool pu, bool data, bool vs