/*-
 * Copyright (c) 2008, 2009 Yahoo!, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The names of the authors may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: releng/11.1/usr.sbin/mfiutil/mfiutil.c 258053 2013-11-12 17:10:56Z sbruno $
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "mfiutil.h"
#include <locale.h>
#ifdef __linux__
#include <sys/sysmacros.h>
#endif

#if defined __FreeBSD__ && !defined __FreeBSD_kernel__
#define __FreeBSD_kernel__
#endif

SET_DECLARE(MFI_DATASET(top), struct mfiutil_command);

MFI_TABLE(top, start);
MFI_TABLE(top, stop);
MFI_TABLE(top, abort);

#ifdef HAVE_MFI_DRIVER
int use_mfi = 1;
#elif defined __sun && defined __SVR4
size_t driver_name_length = 6;
const char *driver_name = "mr_sas";
#endif
int mfi_unit;
int mfi_opts;
static int fw_name_width, fw_version_width, fw_date_width, fw_time_width;

static void
print_usage(const char *name)
{
	fprintf(stderr, "Usage: %s [-dep] "
#ifdef HAVE_MFI_DRIVER
		"[-t {mfi|mrsas}] "
#elif defined __sun && defined __SVR4
		"[-t <drv_type>] "
#endif
		"[-u <unit>] <command> ...\n\n", name);
	fprintf(stderr, "Commands include:\n");
	fprintf(stderr, "    version\n");
	fprintf(stderr, "    show adapter              - display controller information\n");
	fprintf(stderr, "    show battery              - display battery information\n");
	fprintf(stderr, "    show config               - display RAID configuration\n");
	fprintf(stderr, "    show drives               - list physical drives\n");
	fprintf(stderr, "    show events               - display event log\n");
	fprintf(stderr, "    show firmware             - list firmware images\n");
	fprintf(stderr, "    show foreign              - display detected foreign volumes\n");
	fprintf(stderr, "    show logstate             - display event log sequence numbers\n");
	fprintf(stderr, "    show volumes              - list logical volumes\n");
	fprintf(stderr, "    show patrol               - display patrol read status\n");
	fprintf(stderr, "    show progress             - display status of active operations\n");
	fprintf(stderr, "    fail <drive>              - fail a physical drive\n");
	fprintf(stderr, "    good <drive>              - set a failed/SYSPD drive as UNCONFIGURED\n");
	fprintf(stderr, "    rebuild <drive>           - mark failed drive ready for rebuild\n");
	fprintf(stderr, "    syspd <drive>             - set drive into use as non-RAID SYSTEM physical drive\n");
	fprintf(stderr, "    drive progress <drive>    - display status of active operations\n");
	fprintf(stderr, "    drive clear <drive> {start|stop} - clear a drive with all 0x00\n");
	fprintf(stderr, "    start rebuild <drive>\n");
	fprintf(stderr, "    abort rebuild <drive>\n");
	fprintf(stderr, "    locate <drive> {on|off}   - toggle drive LED\n");
	fprintf(stderr, "    cache <volume> [<command> [<setting>]]\n");
	fprintf(stderr, "    name <volume> <name>\n");
	fprintf(stderr, "    volume progress <volume>  - display status of active operations\n");
	fprintf(stderr, "    clear                     - clear volume configuration\n");
	fprintf(stderr, "    create <type> [-v] [-s <stripe_size>] <drive>[,<drive>[,...]] [<drive>[,<drive>[,...]]\n");
	fprintf(stderr, "    delete <volume>\n");
	fprintf(stderr, "    add <drive> [<volume>]    - add a hot spare\n");
	fprintf(stderr, "    remove <drive>            - remove a hot spare\n");
	fprintf(stderr, "    patrol {disable|auto|manual} [<interval> [<start>]]\n");
	fprintf(stderr, "    start patrol              - start a patrol read\n");
	fprintf(stderr, "    stop patrol               - stop a patrol read\n");
	fprintf(stderr, "    foreign scan              - scan for foreign configurations\n");
	fprintf(stderr, "    foreign clear [<volume>]  - clear foreign configurations (default all)\n");
	fprintf(stderr, "    foreign diag [<volume>]   - diagnostic display foreign configurations (default all)\n");
	fprintf(stderr, "    foreign preview [<volume>] - preview foreign configurations (default all)\n");
	fprintf(stderr, "    foreign import [<volume>] - import foreign configurations (default all)\n");
	fprintf(stderr, "    flash <firmware>\n");
	fprintf(stderr, "    start learn               - start a BBU relearn\n");
	fprintf(stderr, "    bbu <setting> <value>     - set BBU properties\n");
	fprintf(stderr, "    ctrlprop rebuild [<rate>] - get/set the volume rebuild rate\n");
	fprintf(stderr, "    ctrlprop alarm [on|off]   - get/set controller alarm status\n");
#ifdef DEBUG
	fprintf(stderr, "    debug                     - debug 'show config'\n");
#if defined HAVE_MFI_DRIVER && (defined __FreeBSD_kernel__ || defined __DragonFly__)
	fprintf(stderr, "    dump                      - display 'saved' config\n");
#endif
#endif
}

static void create_device_node_as_needed() {
#ifdef __linux__
	struct stat st;
	if(stat("/dev/megaraid_sas_ioctl_node", &st) == 0 && S_ISCHR(st.st_mode)) return;
	FILE *f = fopen("/proc/devices", "r");
	if(!f) return;
	int is_chr_dev = 0;
	char buffer[32];
	int len;
	while((len = fgetline(f, buffer, sizeof buffer)) != -1) {
		if(len < 0) {
			int c;
			while((c = fgetc(f)) != '\n') {
				if(c == EOF) {
					fclose(f);
					return;
				}
			}
			continue;
		}
		if(!is_chr_dev) {
			if(strcmp(buffer, "Character devices:") == 0) is_chr_dev = 1;
			continue;
		}
		if(!len) break;
		if(len < 20) continue;
		if(memcmp(buffer + (len - 19), " megaraid_sas_ioctl", 19) == 0) {
			char *end_p;
			unsigned int dev_maj = strtoul(buffer, &end_p, 10);
			if(*end_p != ' ') break;
			unlink("/dev/megaraid_sas_ioctl_node");
			if(mknod("/dev/megaraid_sas_ioctl_node", S_IFCHR | 0600, makedev(dev_maj, 0)) < 0) {
				warn("mknod: /dev/megaraid_sas_ioctl_node");
			}
			break;
		}
	}
	fclose(f);
#elif defined HAVE_MFI_DRIVER && (defined __FreeBSD_kernel__ || defined __DragonFly__)
	if(use_mfi) return;
	struct stat st;
	if(lstat("/dev/megaraid_sas_ioctl_node", &st) == 0) {
		if(S_ISCHR(st.st_mode)) return;
		if(S_ISLNK(st.st_mode)) {
			char target[5];
			int len = readlink("/dev/megaraid_sas_ioctl_node", target, sizeof target);
			if(len > 0) {
				if(len != 4) return;
				if(memcmp(target, "mfi0", 4)) return;
			}
		}
	}
	if(stat("/dev/mrsas0", &st) < 0) return;
	if(!S_ISCHR(st.st_mode)) return;
	unlink("/dev/megaraid_sas_ioctl_node");
	if(symlink("mrsas0", "/dev/megaraid_sas_ioctl_node") < 0) {
		warn("symlink: /dev/megaraid_sas_ioctl_node");
	}
#endif
}

static int
version(int ac __unused, char **av __unused)
{

	fputs("mfiutil version 1.0.15", stdout);
#ifdef LOCAL_VERSION
	fputs(LOCAL_VERSION, stdout);
#endif
#ifdef DEBUG
	fputs(" (DEBUG)", stdout);
#endif
	fputc('\n', stdout);
	return (0);
}
MFI_COMMAND(top, version, version);

int
main(int ac, char **av)
{
	struct mfiutil_command **cmd;
	int ch;

	const char *name = strrchr(av[0], '/');
	if(name) name++; else name = av[0];

#ifdef HAVE_MFI_DRIVER
	if(strcmp(name, "mrsasutil") == 0) use_mfi = 0;
#endif

	setlocale(LC_ALL, "");

	while ((ch = getopt(ac, av, "+dept:u:")) != -1) {
		switch (ch) {
		case 'd':
			mfi_opts |= MFI_DNAME_DEVICE_ID;
			break;
		case 'e':
			mfi_opts |= MFI_DNAME_ES;
			break;
		case 'p':
			mfi_opts |= MFI_PARSABLE_NUMBER;
			break;
		case 't':
#ifdef HAVE_MFI_DRIVER
			if(strcmp(optarg, "mrsas") == 0) use_mfi = 0;
			else if(strcmp(optarg, "mfi") == 0) use_mfi = 1;
			else errx(-1, "Driver type '%s' not recognized", optarg);
#elif defined __sun && defined __SVR4
			driver_name_length = strlen(optarg);
			if(!driver_name_length || driver_name_length > 16) {
				errx(-1, "Invalid driver type specified");
			}
			driver_name = optarg;
#endif
			break;
		case 'u':
			mfi_unit = atoi(optarg);
			if(mfi_unit < 0 || mfi_unit > (int)UINT16_MAX) errx(-1, "Invalid unit specified");
			break;
		case '?':
			print_usage(name);
			return -1;
		}
	}

	av += optind;
	ac -= optind;

	if (ac == 0) {
		print_usage(name);
		return -1;
	}

	create_device_node_as_needed();

	SET_FOREACH(cmd, MFI_DATASET(top)) {
		if (strcmp((*cmd)->name, av[0]) == 0) {
			int e = (*cmd)->handler(ac, av);
			return e ? 1 : 0;
		}
	}
	warnx("Unknown command %s.", av[0]);
	return (1);
}

void
scan_firmware(const struct mfi_info_component *comp)
{
	int len;

	len = strlen(comp->name);
	if (fw_name_width < len)
		fw_name_width = len;
	len = strlen(comp->version);
	if (fw_version_width < len)
		fw_version_width = len;
	len = strlen(comp->build_date);
	if (fw_date_width < len)
		fw_date_width = len;
	len = strlen(comp->build_time);
	if (fw_time_width < len)
		fw_time_width = len;
}

void
display_firmware(const struct mfi_info_component *comp, const char *tag)
{

	printf("%-*s  %-*s  %-*s  %-*s  %s\n", fw_name_width, comp->name,
	    fw_version_width, comp->version, fw_date_width, comp->build_date,
	    fw_time_width, comp->build_time, tag);
}
