#include "aptpackagesearch.h"

#include <QRegularExpression>

#include "runcommandforparsing.h"

#include "ipackagedb.h"
#include "iprogressobserver.h"

namespace NApt {

class AptCacheSearcher {
    const QStringList& includePatterns;
    bool searchDescr;
    bool wholeWords;

    QString createSearchCommandLine() const {
        if (searchDescr) {
            return createSearchInDescriptionCommandLine();
        } else {
            return createSearchInNamesCommandline();
        }
    }

    QString createSearchInDescriptionCommandLine() const {
        QString commandLine = "apt-cache search";
        if (wholeWords) {
            QStringList boundaryRegexes = QStringList(includePatterns.size(), "");
            std::transform(
                includePatterns.begin(),
                includePatterns.end(),
                boundaryRegexes.begin(),
                [](const QString& pattern) {
                    return createMatchBoundaryRegex(pattern);
                }
                );
            commandLine += " " + boundaryRegexes.join(" ");
        } else {
            commandLine += " " + includePatterns.join(" ");
        }
        commandLine += " | awk '{print $1}'";
        return commandLine;
    }

    QString createSearchInNamesCommandline() const {
        QStringList boundaryRegexes = QStringList(includePatterns.size(), "");
        std::transform(
            includePatterns.begin(),
            includePatterns.end(),
            boundaryRegexes.begin(),
            [this](const QString& pattern) {
                if (wholeWords)
                    return "grep -P " + createMatchBoundaryRegex(pattern);
                else
                    return "grep " + pattern;
            }
        );
        return ("apt-cache pkgnames | " + boundaryRegexes.join(" | "));
    }

    static QString createMatchBoundaryRegex(const QString& name) {
        return "\"\\b(\\d*|lib)" + name + "\\d*\\b\"";
    }

public:
    AptCacheSearcher(const QStringList& includePatterns,
                     bool searchDescr,
                     bool wholeWords)
        : includePatterns(includePatterns),
        searchDescr(searchDescr),
        wholeWords(wholeWords) {}


    bool search(set<string>& resultCollector) {
        bool foundAPackage = false;
        auto lineProcessor = [&resultCollector, &foundAPackage](const string& line) {
            foundAPackage = true;
            resultCollector.insert(line);
        };
        QString commandLine = createSearchCommandLine();

        NApplication::runCommandForParsing(commandLine.toStdString(), lineProcessor);
        return foundAPackage;
    }
};

AptPackageSearch::AptPackageSearch(IPackageDB* pPackageDB) :
    _pPackageDB(pPackageDB)
{
}

bool AptPackageSearch::search(std::set<string>& result,
                              const QStringList& includePatterns,
                              bool searchDescr,
                              bool wholeWords) const {
    AptCacheSearcher searcher(includePatterns, searchDescr, wholeWords);
    return searcher.search(result);
};

void AptPackageSearch::reloadPackageInformation(NUtil::IProgressObserver* pObserver) {
    if (pObserver)
        pObserver->setProgress(100);
}

} // namespace NApt
