/*
 * LongTermBatch.java
 *
 * Created on 12. Juli 2007, 14:05
 *
 */
package pauker.program;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * a long term batch
 * @author Ronny.Standtke@gmx.net
 */
public class LongTermBatch extends Batch {

    private final int batchNumber;
    private final Set<Card> expiredCards;
    private final long expirationTime;
    private static final long ONE_SECOND = 1000;
    private static final long ONE_MINUTE = ONE_SECOND * 60;
    private static final long ONE_HOUR = ONE_MINUTE * 60;
    private static final long ONE_DAY = ONE_HOUR * 24;
    private static final long EXPIRATION_UNIT = ONE_DAY;

    /**
     * Creates a new instance of LongTermBatch
     * @param batchNumber the number of this long term batch
     */
    public LongTermBatch(int batchNumber) {
        super(null);
        this.batchNumber = batchNumber;
        double factor = Math.pow(Math.E, batchNumber);
        expirationTime = (long) (EXPIRATION_UNIT * factor);
        expiredCards = new HashSet<Card>();
    }

    /** returns the batch number
     * @return the batch number
     */
    public int getBatchNumber() {
        return batchNumber;
    }

    /** returns the expiration time of this batch
     * @return the expiration time of this batch
     */
    public long getExpirationTime() {
        return expirationTime;
    }

    /** adds a card to this batch
     * @param card the new card
     */
    @Override
    public void addCard(Card card) {
        card.setLongTermBatchNumber(batchNumber);
        card.setExpirationTime(expirationTime);
        cards.add(card);
    }

    /** removes a card from the batch
     * @param card the card to be removed
     */
    @Override
    public void removeCard(Card card) {
        expiredCards.remove(card);
        super.removeCard(card);
    }

    /**
     * removes a card from the batch
     * @param index the index where the card should be removed
     * @return the removed card
     */
    @Override
    public Card removeCard(int index) {
        Card card = super.removeCard(index);
        expiredCards.remove(card);
        return card;
    }

    /**
     * returns a list of all expired cards of this batch
     * @return a list of all expired cards of this batch
     */
    public Set<Card> getExpiredCards() {
        return expiredCards;
    }

    /** returns the number of expired cards
     * @return the number of expired cards
     */
    public int getNumberOfExpiredCards() {
        return expiredCards.size();
    }

    /** gets an expired card
     * @param i the index of the expired card
     * @return the expired card
     */
    public Card getOldestExpiredCard() {
        // return the card with the oldest expiration date
        Iterator iterator = expiredCards.iterator();
        Card oldestCard = (Card) iterator.next();
        while (iterator.hasNext()) {
            Card tmpCard = (Card) iterator.next();
            if (tmpCard.getExpirationTime() < oldestCard.getExpirationTime()) {
                oldestCard = tmpCard;
            }
        }
        return oldestCard;
    }

    /** recalculates the batch of expired cards */
    public void refreshExpiration() {
        expiredCards.clear();
        long currentTime = System.currentTimeMillis();
        for (Card card : cards) {
            long learnedTime = card.getLearnedTimestamp();
            long diff = currentTime - learnedTime;
            if (diff > expirationTime) {
                expiredCards.add(card);
            }
        }
    }
}
