/*
 * getty.c	This is the part that talks with the modem,
 *		does the login stuff etc.
 *
 * Version:	@(#)getty.c  1.34  13-Jan-1998  MvS.
 *
 */
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <errno.h>

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "server.h"

int tc_getattr(int fd, struct termios *termios_p)
{
  if(tcgetattr(fd, termios_p))
  {
    nsyslog(LOG_ERR, "Can't get serial attributes:%m");
    return -1;
  }
  return 0;
}

int tc_setattr(int fd, int optional_actions, struct termios *termios_p)
{
  if(tcsetattr(fd, optional_actions, termios_p))
  {
    nsyslog(LOG_ERR, "Can't set serial attributes:%m");
    return -1;
  }
  return 0;
}

/*
 *	Reset the terminal to a reasonably sane state.
 */
void maketermsane()
{
	struct termios tty;

	tcflush(0, TCIFLUSH);
	tcflush(1, TCOFLUSH);
	tcflush(2, TCOFLUSH);
	if(tc_getattr(0, &tty))
		return;
	tty.c_iflag &= ~IGNCR;
	tty.c_oflag |= OPOST|ONLCR;
	tty.c_lflag |= ISIG|IEXTEN|ICANON|ECHO|ECHOE|ECHOK|ECHOCTL|ECHOKE;
	tc_setattr(0, TCSANOW, &tty);
}

/*
 *	Get the lockfile.
 */
int lockfile(char **lock_name, const char *lockdir, const char *tty
	, LOCATION location)
{
	struct stat st;
	char *p; /* tmp variable for parsing */
	char buf[MAXPATHLEN + 1];

	/*
	 *	First some checks to see if we do want to
	 *	use a lockdir and if it exists.
	 */
	if(lockdir == NULL || lockdir[0] != '/')
	{
		*lock_name = NULL;
		return -1;
	}
	if(stat(lockdir, &st) < 0 || !S_ISDIR(st.st_mode))
	{
		*lock_name = NULL;
		return -1;
	}
	if((p = strrchr(tty, '/')) != NULL)
		tty = p + 1;
	if(snprintf(buf, MAXPATHLEN, "%s/%s..%s"
		, lockdir, (location == eLocBuf ? "BUF" : "LCK"), tty) == -1)
	{
		*lock_name = NULL;
		return (-1);
	}
	*lock_name = xstrdup(buf);
	return 0;
}


/*
 *	See if a device is locked.
 */
int islocked(const char *lock_name)
{
	FILE *fp;
	int pid;

	if(!*lock_name)
		return 0;
	if((fp = fopen(lock_name, "r")) == NULL)
		return 0;
	if(fscanf(fp, "%d", &pid) != 1 || (kill(pid, 0) < 0 && errno == ESRCH))
	{
		fclose(fp);
		if(unlink(lock_name))
		{
			nsyslog(LOG_CRIT, "Can't remove stale lockfile \"%s\".", lock_name);
			xsleep(30);
		}
		return 0;
	}
	fclose(fp);
	return pid;
}

/*
 *	Create a lockfile.
 *	Returns: -1   some error
 *			0   success
 *			1   already locked.
 */
int dolock(const char *lock_name)
{
	char tmp[MAXPATHLEN+1];
	FILE *fp;
	int i;

	if (!*lock_name)
		return 0;

	snprintf(tmp, sizeof (tmp), "%s.%d", lock_name, getpid ());
	unlink(tmp);

	/*
	 *	Already locked?
	 */
	if(islocked(lock_name))
		return 1;
	/*
	 *	Try to lock.
	 */
	if((fp = fopen(tmp, "w")) == NULL)
	{
		nsyslog(LOG_ERR, "%s: %m", tmp);
		return -1;
	}
	fprintf(fp, "%10d portslave root\n", getpid());
	fclose(fp);

	for(i = 0; i < 5; i++)
	{
		if(i > 0)
			xsleep(5);
		islocked(lock_name); /* removes the file if no process */
		if(link(tmp, lock_name) == 0) break;
	}
	unlink(tmp);

	return (i == 5) ? 1 : 0;
}

/* repeatedly try to create a lock, wait sleep_time seconds between attempts */
void dolock_and_wait(int sleep_time)
{
  while(dolock(lineconf.lockfile) == 1)
  {
    while(islocked(lineconf.lockfile))
      xsleep(sleep_time);
  }
}

#ifdef CYCLADES
static char *term_in_use(LOCATION location)
{
	if(!lineconf.lockfile)
		lockfile(&lineconf.lockfile, lineconf.lockdir, lineconf.tty, location);

#ifndef CYCLADES
	dolock_and_wait(10);
#else
	if(dolock(lineconf.lockfile) == 1 && islocked(lineconf.lockfile))
	{
		snprintf(buf, sizeof(buf), 
			"\r\n*\r\n* * * %s is being used !!!\r\n*\r\n",
			lc->tty);
		return(buf);
	}
#endif
	return (NULL);
}
#endif

/*
 *  Close the serial line
 */
void serial_close()
{
	maketermsane();
	close(0);
	close(1);
	close(2);
}

/*
 *	Open the serial line and put it into raw/no echo mode.
 */
int serial_open(LOCATION location)
{
	int fd, fl;
	struct termios tio, old;
	char *tty = lineconf.tty;
	tcflag_t PDataSize;
	speed_t Speed;
	int MLines;

	if(tty == NULL || tty[0] == 0)
	{
		nsyslog(LOG_ERR, "no tty specified");
		return -1;
	}
	if(tty[0] != '/')
	{
		nsyslog(LOG_ERR, "Coding error, tty should be canonical");
		exit(1);
	}

	if((fd = open(tty, O_RDWR|O_NONBLOCK|O_NOCTTY)) < 0)
	{
		nsyslog(LOG_ERR, "%s: %m", tty);
		return -1;
	}

	if(location == eLocModem)
	{
/* Toggle DTR in case of modem hang */
		if(tc_getattr(fd, &tio) || tc_getattr(fd, &old) )
			return -1;
		cfsetospeed(&tio, B0);
		cfsetispeed(&tio, B0);
		if(tc_setattr(fd, TCSANOW, &tio))
			return -1;
		xusleep(500000);
		if(tc_setattr(fd, TCSANOW, &old))
			return -1;
/* REALLY Make sure, toggle it again */
		if(tc_setattr(fd, TCSANOW, &tio))
			return -1;
		xusleep(500000);
		if(tc_setattr(fd, TCSANOW, &old))
			return -1;

		fl = fcntl(fd, F_GETFL, 0);
		fl &= ~O_NONBLOCK;
		fcntl(fd, F_SETFL, fl);

		xusleep(500000);
		if(tc_setattr(fd, TCSANOW, &old))
			return -1;
	}

	ioctl(fd,TIOCMGET,&MLines);
	MLines = MLines | TIOCM_DTR;
	ioctl(fd,TIOCMSET,&MLines);

	if(tc_getattr(fd, &tio))
		return -1;
	tio.c_iflag = IGNPAR|ICRNL|IGNBRK;
	tio.c_oflag = OPOST|ONLCR;
	tio.c_lflag = 0; /* ICANON|ECHO|ECHONL*/

	if(location != eLocModem)
		tio.c_cflag = CREAD|CS8;
	else
		tio.c_cflag = CREAD|CS8|HUPCL;
	if(location == eLocBuf)
		tio.c_cflag |= CLOCAL;
	else
		if(!lineconf.dcd) tio.c_cflag |= CLOCAL;

	switch (lineconf.flow) {
		case FLOW_NONE:
			break;
		case FLOW_HARD:
			tio.c_cflag |= CRTSCTS;
			break;
		case FLOW_SOFT:
			tio.c_iflag |= IXON|IXOFF;
			tio.c_cc[VSTART] = 17;
			tio.c_cc[VSTOP]  = 19;
			break;
	}

	switch(lineconf.speed) {
		case 1200:
			Speed = B1200;
			break;
		case 2400:
			Speed = B2400;
			break;
		case 4800:
			Speed = B4800;
			break;
		case 9600:
			Speed = B9600;
			break;
		case 19200:
			Speed = B19200;
			break;
		case 38400:
			Speed = B38400;
			break;
		case 57600:
			Speed = B57600;
			break;
		case 115200:
			Speed = B115200;
			break;
		case 230400:
			Speed = B230400;
			break;
		case 460800:
			Speed = B460800;
			break;
		case 921600:
			Speed = B921600;
			break;
		default:
			nsyslog(LOG_WARNING, "%d: line speed not supported (using 9600)",
				lineconf.speed);
			Speed = B9600;
			break;
	}

	switch(lineconf.stopbits)
	{
		case 2:
			tio.c_cflag = tio.c_cflag | CSTOPB;
			break;
		case 1:
		default:
			/* there is no support to 1.5 stop bits defaults to 1 */
			tio.c_cflag = tio.c_cflag & ~CSTOPB;
			break;
	}

	switch(lineconf.parity)
	{
		case 2:
			tio.c_cflag = tio.c_cflag | PARENB | PARODD;
			break;
		case 3:
			tio.c_cflag = (tio.c_cflag | PARENB) & ~PARODD;
			break;
		case 1:
		default:
			/* There's no support for MARK/SPACE parity so sets no parity */
			tio.c_cflag = tio.c_cflag & ~PARENB;
			break;
	}

	switch(lineconf.datasize)
	{
		case 5:
			PDataSize = CS5;
			break;
		case 6:
			PDataSize = CS6;
			break;
		case 7:
			PDataSize = CS7;
			break;
		case 8:
		default:
			PDataSize = CS8;
			break;
	}
	tio.c_cflag = tio.c_cflag & ((tio.c_cflag & ~CSIZE) | PDataSize);

	/* Setting speed above 38400 correctly */
	cfsetospeed(&tio, Speed);
	cfsetispeed(&tio, Speed);

	for(fl=0; fl < 3 && tcsetattr(fd, TCSANOW, &tio) != 0; fl++)
	{
		if(fl == 2)
		{
			nsyslog(LOG_ERR, "tcsetattr %s: %m", tty);
			return -1;
		}
		xsleep(1);
	}

	if(lineconf.protocol != P_SOCKET_SERVER && lineconf.protocol != P_SOCKET_SSH)
	{
		dup2(fd, 0);
		dup2(fd, 1);
		dup2(fd, 2);

		if(fd > 2)
		{
			close(fd);
			fd = 0;
		}
	}

	tcflush(fd, TCIOFLUSH);

	return fd;
}
